function [x,Ax,func,ell2_pppa,ell1_pppa,ell0_pppa,alpha_pppa] = bpdn_pppa(P,tau,nbIter,alpha1,type,lambda,Anorm)

% [x,Ax,func,ell2_pppa,ell1_pppa,ell0_pppa,alpha_pppa] = bp_pppa_bis(P,tau,nbIter,alpha1,type,lambda,Anorm)
%
% Finds, wiht the PPPA algorithm, a solution to 
%                 min    sum(abs( x ))
%          subject to    norm(P.A*x - P.b) <= tau * sqrt(length(b(:)))
% where P is at the SPARCO format
%
% P.b : is the image contained in the structure P.b
% tau : is a non-negative number
% P.A represents a matrix : the multiplication A*x is done by P.A(x,1)
%                 the multiplication A'*c is done by P.A(c,2)              '
%                 the cell array containing the size of the matrix 
%                 is obtained with P.A(c,0) 
%
% nbIter : is the number of iterations
% alpha1 is a non negatif double : A smaller alpha provides a better convergence 
%                                 for the norm of the residual. A larger alpha 
%                                 provides a better convergence for the regularity term.
%                                 alpha linearly varies between alpha1 and alpha2.
%
%                                 Depending on 'type', alpha1 can be an array 
% type : 1, 2 or 3 :   if type==1: Alpha = alpha1(1), all along the iterative process
%		       if type==2: AlphaMin=alpha1(1) and AlphaMax=alpha1(2) 
%                                   see paper for the description of AlphaMin and AlphaMax
%		       if type==3: Alpha evolves and is initialised with alpha1(1)
%
% lambda : is an input parameter that is used to compute the statistic func 
%          (value of the functional sum(abs( x )) + lambda norm(P.A*x - P.b) )
%          It is not used by the solver.
%
% Anorm : is operator norm of A (it should be computed offline with the function get_operator_norm)
%
%
% The results are :
% x : the coordinates
% Ax : the reconstruction from the coordinates
% func : a signal containing  norm(P.A*x - P.b) + lambda * sum(abs( x )) (for comparizon purpose)
% ell2_pppa : a signal containing  sqrt( mean( (P.A*x - P.b).^2 )), along the iterative process
% ell1_pppa : a signal containing   mean( fabs( x ) ), along the iterative process
% ell0_pppa : a signal containing  100*mean( x~=0 ), along the iterative process
% alpha_pppa : the value of alpha along the iterative process


% --------------------------------- test some inputs  ----------------------------------

if lambda <= 0,
    fprintf('lambda should be non negative \n');
    return;
end;

if tau <= 0,
    fprintf('tau should be non negative \n');
    return;
end;

if nbIter < 2,
    fprintf('number of iteration must be larger than 2 \n');
    return;
end;

if type ~= 1 && type ~=2 && type ~=3,
   fprintf('type should be 1, 2 or 3 \n');
   return;
end;



% ----------------------------------- Run  PPPA --------------------------------------------


%%%%%%%%%%%%%%%%%%%%%%%%%% initialization 
% The variable names correspond to the notation of the paper

b = P.b(:);
siz= P.A(P.b,0);            
sizeP=cell2mat(siz(1,2));
sizeN=cell2mat(siz(1,1));
rho_without_alpha=1./(Anorm*Anorm);   
target_mse=tau;
tau=tau*sqrt(sizeN);
x = zeros(sizeP,1);
cm = init_c(x,b,P);

if type==1,
  Alpha = alpha1(1);
elseif type==2,
  Alpha = alpha1(1)+alpha1(2)/100;
  rap_alpha = 1.001;
else
  Alpha = alpha1(1);
end;
safety_alpha=25;

[dk,residual_norm,Ax] = compute_dk(P,b,x,cm,Alpha,tau);
oldAx=Ax;

% few stats
func(1) = sum( (oldAx-b).^2 ) + lambda * sum(abs(x));
ell0_pppa(1) = 100 * sum( abs(x)>1e-10 ) / length(x);
ell1_pppa(1) = sum( abs(x) ) / length(x);  
ell2_pppa(1) = residual_norm;   
alpha_pppa(1)=Alpha;


%----------------- main loop 

time=1;
m=0;

while (time < nbIter),  %loop in m
   k=0;
   normEvol=1.;   

   while  ( normEvol > 10/128 & k<50 & time<nbIter), % loop in k
     Atdk = P.A(dk,2); 

     x = update_coordinates(x,rho_without_alpha*Alpha,Atdk);
     
     %%% initialisation of the next iteration 
     %%% (it is done here in order to compute the correct residual_norm in the stats)
 
     [dk,residual_norm,Ax] = compute_dk(P,b,x,cm,Alpha,tau);

     normEvol=sqrt(mean( (Ax-oldAx).^2 )) ;

     %%% computes stats
     func(time+1) = sum((Ax-b).^2) + lambda*sum(abs(x));
     ell0_pppa(time+1) = 100*sum(abs(x)>1e-10)/length(x);
     ell1_pppa(time+1) = sum(abs(x))/length(x);  
     ell2_pppa(time+1) = residual_norm;          
     alpha_pppa(time+1) = Alpha;          

     fprintf('Time : %i m: %i  k: %i : alpha : %g [L2,L1,L0]:  %10.5f %10.5f %10.5f  \n',time,m,k,Alpha,ell2_pppa(time+1),ell1_pppa(time+1),ell0_pppa(time+1));

     %%% end of stats calculus
     
     oldAx=Ax;

     k=k+1;
     time=time+1;
     safety_alpha=safety_alpha+1;   
   end;            % end loop in k

   %% computes next Alpha

   if type==2,
      if residual_norm > target_mse * 1.1,
          Alpha = max(Alpha/rap_alpha,alpha1(1));
      elseif residual_norm < target_mse/1.1,
          Alpha = min(Alpha*rap_alpha,alpha1(2));
      end;
   elseif type==3,
      if residual_norm <= target_mse && safety_alpha>25,
          Alpha=2*Alpha; 
          safety_alpha=0;       
       end;
   end;

   m=m+1;
   cm=dk;
end;     % end loop in m
